/****************************************************************************
 *   $Id:: adc.c 3635 2010-06-02 00:31:46Z usb00423                         $
 *   Project: NXP LPC11xx ADC example
 *
 *   Description:
 *     This file contains ADC code example which include ADC 
 *     initialization, ADC interrupt handler, and APIs for ADC
 *     reading.
 *
 ****************************************************************************
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.
****************************************************************************/
#include "LPC11xx.h"			/* LPC11xx Peripheral Registers */
#include "adc.h"


/*****************************************************************************
** Function name:		ADCInit
**
** Descriptions:		initialize ADC channel
**
** parameters:			ADC clock rate
** Returned value:		None
** 
*****************************************************************************/
void ADCInit( uint32_t ADC_Clk )
{

  /* Disable Power down bit to the ADC block. */  
  LPC_SYSCON->PDRUNCFG &= ~(0x1<<4);

  /* Enable AHB clock to the ADC. */
  LPC_SYSCON->SYSAHBCLKCTRL |= (1<<13);

  LPC_IOCON->PIO1_4   = 0x41; /* Change PIO1_4 (AD5) to analog mode, with no pull-up  */	
  LPC_IOCON->PIO1_10  = 0x41; /* Change PIO1_10 (AD6) to analog mode, with no pull-up */	
  LPC_IOCON->PIO1_11  = 0x41; /* Change PIO1_11 (AD7) to analog mode, with no pull-up */	

  LPC_ADC->CR = ( 0x01 << 0 ) |  /* SEL=1,select channel 0~7 on ADC0 */
	( ( SystemAHBFrequency / ADC_Clk - 1 ) << 8 ) |  /* CLKDIV = Fpclk / 1000000 - 1 */ 
	( 0 << 16 ) | 		/* BURST = 0, no BURST, software controlled */
	( 0 << 17 ) |  		/* CLKS = 0, 11 clocks/10 bits */
	( 1 << 21 ) |  		/* PDN = 1, normal operation */
	( 0 << 22 ) |  		/* TEST1:0 = 00 */
	( 0 << 24 ) |  		/* START = 0 A/D conversion stops */
	( 0 << 27 );		/* EDGE = 0 (CAP/MAT singal falling,trigger A/D conversion) */ 

  return;
}

/********************************************************************************
** Function name:		MeasureCurrent
**
** Descriptions:		Read ADC5 which is the voltage across a 0.75 ohm resistor
**
** parameters:			None
** Returned value:		The average of 4 measurements
** 
*********************************************************************************/
uint32_t MeasureCurrent( void )
{

  uint32_t regVal, ADC_Data, i;

  ADC_Data = 0;

  for(i=0; i<4; i++) {

  LPC_ADC->CR &= 0xFFFFFF00;
  LPC_ADC->CR |= (1 << 24) | (1 << 5);	 /* channel 5, start A/D convert */

  while ( 1 ) { /* wait until end of A/D convert */
  
	regVal = *(volatile unsigned long *)(LPC_ADC_BASE + ADC_OFFSET + ADC_INDEX * 5);
	
	if ( regVal & ADC_DONE ) { /* read result of A/D conversion */
	  break;
	}
  }	
        
  LPC_ADC->CR &= 0xF8FFFFFF;	/* stop ADC now */    

  ADC_Data += ( ( regVal >> 6 ) & 0x3FF );
  }

  return ( ADC_Data / 4 );	/* return average of four A/D conversions */

}

/*****************************************************************************
** Function name:		MeasureVBAT
**
** Descriptions:		Measures the average battery voltage
** 						The voltage is measured 4 times and the average
**						value is returned
**
** parameters:			None
** Returned value:		The average of four battery measurements
** 
*****************************************************************************/
uint32_t MeasureVBAT( void )
{

uint32_t regVal, i;
uint32_t ADC5 = 0;
uint32_t ADC7 = 0;

for(i=0; i<4; i++) {
/*************** Measure ADC7 **********************************/
  LPC_ADC->CR &= 0xFFFFFF00;
  LPC_ADC->CR |= (1 << 24) | (1 << 7);	 /* switch channel,start A/D convert */

  while ( 1 ) { /* wait until end of A/D convert */
  
	regVal = *(volatile unsigned long *)(LPC_ADC_BASE + ADC_OFFSET + ADC_INDEX * 7);
	
	if ( regVal & ADC_DONE ) { /* read result of A/D conversion */
	  break;
	}
  }	
        
  LPC_ADC->CR &= 0xF8FFFFFF;	/* stop ADC now */    

  ADC7 += ( ( regVal >> 6 ) & 0x3FF );
/*************** End Measure ADC7 ******************************/

/*************** Measure ADC5 **********************************/
  LPC_ADC->CR &= 0xFFFFFF00;
  LPC_ADC->CR |= (1 << 24) | (1 << 5);	 /* switch channel,start A/D convert */

  while ( 1 ) { /* wait until end of A/D convert */
  
	regVal = *(volatile unsigned long *)(LPC_ADC_BASE + ADC_OFFSET + ADC_INDEX * 5);
	
	if ( regVal & ADC_DONE ) { /* read result of A/D conversion */
	  break;
	}
  }	
        
  LPC_ADC->CR &= 0xF8FFFFFF;	/* stop ADC now */    

  ADC5 += ( ( regVal >> 6 ) & 0x3FF );

/*************** End Measure ADC5 ******************************/

}
  
ADC7 /= 4;
ADC5 /= 4;
return (ADC7 * 5 / 3) - ADC5;
}

/*********************************************************************************
**                            End Of File
*********************************************************************************/
